#include <iostream>
#include <cmath>

class Rational
{
public:
  Rational();
  Rational(int);
  Rational(int, int);

  int denom()const;
  int num()const;

  double value()const;
  operator double()const;

  const Rational& operator+=(const Rational& b);
  const Rational& operator-=(const Rational& b);
  const Rational& operator*=(const Rational& b);
  const Rational& operator/=(const Rational& b);

private:
  int m_denom;
  int m_num;

  void reduce();

  friend std::ostream& operator<<(std::ostream& os, const Rational& r);
  friend std::istream& operator>>(std::istream& is, Rational& r);
};

// Construct a Rational undefined - no default value
Rational::Rational()
{
}

// Construct a Rational
Rational::Rational(int n, int d)
{
  m_num = n;
  m_denom = d;
  if(d == 0)
  {
    std::cerr << "Attempt to define a Rational with zero denominator" << std::endl;
  }
}

// Convert an integer into a rational number
Rational::Rational(int n)
{
  m_num = n;
  m_denom = 1;
}

int Rational::num()const
{
  return m_num;
}

int Rational::denom()const
{
  return m_denom;
}

// Return the approximate value of this rational number
double Rational::value()const
{
  return (double)m_num / (double)m_denom;
}

// Allow casting to a double.
Rational::operator double()const
{
  return value();
}


/* Now follow the operate-and-assign overloaded operators.

   These take a single parameter (the right-hand side),
   modify the current Rational, and then return the modified Rational.
   This is done as a const &.
 */

// Add and assign
const Rational& Rational::operator+=(const Rational& b)
{
  Rational c;
  c.m_num = m_num * b.m_denom + b.m_num * m_denom;
  c.m_denom = m_denom * b.m_denom;
  c.reduce();

  m_num = c.m_num;
  m_denom = c.m_denom;

  return *this;
}

// Subtract and assign
const Rational& Rational::operator-=(const Rational& b)
{
  Rational c;
  c.m_num = m_num * b.m_denom - b.m_num * m_denom;
  c.m_denom = m_denom * b.m_denom;
  c.reduce();

  m_num = c.m_num;
  m_denom = c.m_denom;

  return *this;
}

// Multiply and assign
const Rational& Rational::operator*=(const Rational& b)
{
  Rational c;
  c.m_num = m_num * b.m_num;
  c.m_denom = m_denom * b.m_denom;
  c.reduce();

  m_num = c.m_num;
  m_denom = c.m_denom;

  return *this;
}

// Divide and assign
const Rational& Rational::operator/=(const Rational& b)
{
  Rational c;
  c.m_num = m_num * b.m_denom;
  c.m_denom = m_denom * b.m_num;
  c.reduce();

  m_num = c.m_num;
  m_denom = c.m_denom;

  return *this;
}

// Reduce a Rational to its simplest form
void Rational::reduce()
{
  // Find the Greatest Common Divisor of two integers:
  int r1, r2;
  if(std::abs(m_num) > std::abs(m_denom))
  {
    r1 = m_num;
    r2 = m_denom;
  }
  else
  {
    r1 = m_denom;
    r2 = m_num;
  }
  
  while(r2 != 0)
  {
    int t = r2;
    r2 = r1 % r2;
    r1 = t;
  }
  
  m_num /= r1;
  m_denom /= r1;
}


/* Binary operators on Rationals.
   These are implemented in terms of the operate-and-assign operators.
   This reduces the likelihood that we make mistakes in copy-pasting.

   They return an object which is not done by reference, since the local variable c
   is destroyed as soon as the function exits.

   Note that binary operator-overloadings should be defined as global functions,
   i.e. outside of the class itself.
 */

Rational operator+(const Rational& a, const Rational& b)
{
  Rational c = a;
  c += b;
  return c;
}

Rational operator-(const Rational& a, const Rational& b)
{
  Rational c = a;
  c -= b;
  return c;
}

Rational operator*(const Rational& a, const Rational& b)
{
  Rational c = a;
  c *= b;
  return c;
}

Rational operator/(const Rational& a, const Rational& b)
{
  Rational c = a;
  c /= b;
  return c;
}

/* Since it is possible for a Rational to be cast to a double,
   we also need operators for the case such as b * 3.

   Here, it is ambiguous whether we expect b to be cast to a double,
   and the operation (double) * (int) be carried out,
   or whether we expect the integer 3 to be cast to a Rational and
   the operation (Rational) * (Rational) to be carried out.

   This is ambiguous according to the C++ standard unless
   we define explicit operators for the case
   whether the RHS of the operator is an integer.
   We should also do this for the case where the LHS is
   an integer - left as a (trivial) exercise for the reader.

   Note that within the functions, (Rational) += (int) (etc.) is well-defined.
 */

Rational operator+(const Rational& a, const int& b)
{
  Rational c = a;
  c += b;
  return c;
}

Rational operator-(const Rational& a, const int& b)
{
  Rational c = a;
  c -= b;
  return c;
}

Rational operator*(const Rational& a, const int& b)
{
  Rational c = a;
  c *= b;
  return c;
}

Rational operator/(const Rational& a, const int& b)
{
  Rational c = a;
  c /= b;
  return c;
}


std::ostream& operator<<(std::ostream& os, const Rational& r)
{
  os << r.num() << "/" << r.denom();
  return os;
}

std::istream& operator>>(std::istream& is, Rational& r)
{
  is >> r.m_num;
  char c;
  is >> c;
  if( c != '/')
  {
    std::cerr << "Ill-defined rational input" << std::endl;
  }
  
  is >> r.m_denom;
  r.reduce();
  return is;
}


int main(void)
{
  Rational a(10,3);
  std::cout << "a = " << a.num() << "/" << a.denom() << " ~ " << a.value() << std::endl;
  a /= 5;
  Rational b(3,4);
  a += b*3;
  std::cout << "a = " << a.num() << "/" << a.denom() << " ~ " << a.value() << std::endl;

  Rational c;
  std::cout << "Please enter a rational number in the form a/b: ";
  std::cin >> c;
  std::cout << "c = " << c << " ~ " << (double)c << std::endl;
  c -= 5;
  std::cout << "c = " << c << " ~ " << (double)c << std::endl;

  c -= b*a;
  std::cout << "c = " << c << " ~ " << (double)c << std::endl;

  return 0;
}

